<?php
declare(strict_types=1);

namespace App\Controller;

use App\Controller\AppController;
use Cake\I18n\FrozenTime;
use Cake\Event\EventInterface;
use Cake\Core\Configure;
use Cake\Utility\Inflector;
use Cake\Log\Log;
use Cake\Filesystem\File;
use Cake\Http\Exception\BadRequestException;
/**
 * Contents Controller
 *
 * @method \App\Model\Entity\Content[]|\Cake\Datasource\ResultSetInterface paginate($object = null, array $settings = [])
 */
class ContentsController extends AppController
{
    public function initialize(): void
    {
        parent::initialize();
        $this->contents = $this->fetchTable('Contents');
        $this->contentImages = $this->fetchTable('ContentImages');
        $this->menus = $this->fetchTable('Menus');
        $this->pageContents = $this->fetchTable('PageContents');
    }

    public function beforeRender(EventInterface $event)
    {
        parent::beforeRender($event);
    }

    public function isAuthorized($user)
    {
        $action = $this->request->getParam('action');
        return true;
    }

    /**
     * Index method
     *
     * @return \Cake\Http\Response|null|void Renders view
     */
    public function index()
    {
        $pageTitle = 'All Contents';
        $action = $this->request->getParam('action');
        $searchTerm = $this->request->getQuery('search');
        $currentMenu = $this->menus->getMenuBySlug($this->slug);
        $relatedMenusWithContents = $this->pageContents->getRelatedMenusWithContents($currentMenu->id);

        $query = $this->searchFilters($relatedMenusWithContents);

        $listings = $this->paginate($query);
        $this->set(compact('pageTitle', 'listings', 'searchTerm'));
    }

    public function allListings($id = null)
    {
        $pageTitle = 'All Contents';
        $searchTerm = $this->request->getQuery('search');
        $currentMenu = $this->menus->getMenuBySlug($this->slug);
        $relatedMenusWithContents = $this->pageContents->getRelatedMenusWithContents($currentMenu->id);
        $query = $this->searchFilters($relatedMenusWithContents);

        $listings = $this->paginate($query);
        $this->set(compact('pageTitle', 'listings', 'searchTerm'));
    }

    public function pendingListings($id = null)
    {
        $pageTitle = 'Pending Contents';
        $action = $this->request->getParam('action');
        $menuSlug = $this->slug;
        $slug = (string) $menuSlug;
        $activeMenu = $this->menus->getMenuBySlug($slug);
        if (!$activeMenu) {
            Log::warning("Menu not found for slug: $slug");
            throw new NotFoundException(__('Menu not found for the given slug.'));
        }

        $query = $this->pageContents->findPendingListings($slug, $activeMenu->id);
        $searchTerm = $this->request->getQuery('search');
        $query = $this->searchFilters($query);

        // Paginate the listings
        $listings = $this->paginate($query);
        $this->set(compact('pageTitle', 'listings', 'searchTerm', 'action'));
    }

    /**
     * Enable listings based on the provided slug.
     *
     * @param int|null $id Optional parameter for additional filtering
     * @return \Cake\Http\Response|null Renders the view
     */
    public function enableListings($id = null)
    {
        $pageTitle = 'Enabled Contents';
        $action = $this->request->getParam('action');
        $menuSlug = $this->slug;
        $slug = (string) $menuSlug;
        $activeMenu = $this->menus->getMenuBySlug($slug);

        if (!$activeMenu) {
            Log::warning("Menu not found for slug: $slug");
            throw new NotFoundException(__('Menu not found for the given slug.'));
        }

        $query = $this->pageContents->findEnableListings($slug, $activeMenu->id);
        $searchTerm = $this->request->getQuery('search');
        $query = $this->searchFilters($query);

        // Paginate the listings
        $listings = $this->paginate($query);
        $this->set(compact('pageTitle', 'listings', 'action'));
    }

    public function disableListings($id = null)
    {
        $pageTitle = 'Disabled Contents';
        $action = $this->request->getParam('action');
        $menuSlug = $this->slug;
        $slug = (string) $menuSlug;
        $activeMenu = $this->menus->getMenuBySlug($slug);

        if (!$activeMenu) {
            Log::warning("Menu not found for slug: $slug");
            throw new NotFoundException(__('Menu not found for the given slug.'));
        }

        $query = $this->pageContents->findDisableListings($slug, $activeMenu->id);
        $searchTerm = $this->request->getQuery('search');
        $query = $this->searchFilters($query);


        // Paginate the listings
        $listings = $this->paginate($query);
        $this->set(compact('pageTitle', 'listings', 'action'));
    }

    /**
     * List edited contents based on the provided slug.
     *
     * @param int|null $id Optional parameter for additional filtering
     * @return \Cake\Http\Response|null Renders the view
     */
    public function editedListings($id = null)
    {
        $pageTitle = 'Edited Contents';
        $action = $this->request->getParam('action');
        $menuSlug = $this->slug;
        $slug = (string) $menuSlug;
        $activeMenu = $this->menus->getMenuBySlug($slug);

        if (!$activeMenu) {
            Log::warning("Menu not found for slug: $slug");
            throw new NotFoundException(__('Menu not found for the given slug.'));
        }

        $query = $this->pageContents->findEditedListings($slug, $activeMenu->id);
        $searchTerm = $this->request->getQuery('search');
        $query = $this->searchFilters($query);

        // Paginate the listings
        $listings = $this->paginate($query);
        $this->set(compact('pageTitle', 'listings', 'action'));
    }

    /**
     * Add method
     *
     * @return \Cake\Http\Response|null|void Redirects on successful add, renders view otherwise.
     */
    public function add($id = null)
    {
        $pageTitle = 'Add Content';
        $images = [];
        $tagId = "";

        $id = (int) $id;
        $parentMenus = $this->menus->getSubmenuOptions($id);

        $content = $this->contents->newEmptyEntity();

        if ($this->request->is(['post'])) {
            Log::info('post request from add form started');

            $data = $this->request->getData();

            if (!empty($data['sbmenu_id'])) {
                $data['menu_id'] = $data['sbmenu_id'];
            } elseif (!empty($data['sbsbmenu_id'])) {
                $data['menu_id'] = $data['sbsbmenu_id'];
            } elseif (!empty($data['sbsbsbmenu_id'])) {
                $data['menu_id'] = $data['sbsbsbmenu_id'];
            } else {
                $data['menu_id'] = $data['pmenu_id'];
            }

            $data["parent_menu_id"] = $data['pmenu_id'];

            $rawHtml = $this->request->getData('content');

            $structuredJson = $this->contents->parseHtmlToBlocksSmart($rawHtml);

            $content = $this->contents->patchEntity($content, $data);

            $content->json_content = $structuredJson;
            Log::info("saving form data: $content");

            if ($this->contents->save($content)) {
                Log::info('form data saved successfully');

                $pagecontent = $this->pageContents->newEmptyEntity();
                $data["content_id"] = $content->id;
                $pc = $this->pageContents->find('All')
                    ->where([
                        "menu_id" => $data["menu_id"],
                        "content_id" => $data["content_id"]
                    ]);

                if (empty($pc->toArray())) {
                    $pagecontent = $this->pageContents->patchEntity($pagecontent, $data);
                    $this->pageContents->save($pagecontent);
                }

                if (isset($data['form_key']) && $data['form_key'] < 0) {
                    $this->contentImages->updateAll(
                        ['content_id' => $content->id],
                        [
                            'content_id' => $data['form_key']
                        ]
                    );
                }

                $this->Flash->success(__('Content added successfully.'));
                $section = $this->request->getParam('section');

                return $this->redirect([
                    'controller' => $section ? $this->request->getParam('section') :
                        $this->request->getParam('controller'),
                    'action' => 'pendingListings'
                ]);

            } else {
                $this->Flash->error(__('The row could not be save. Try again'));
            }
        }

        $menu_id = $id;
        $this->set(compact('parentMenus', 'content', 'images', 'menu_id', 'pageTitle'));
    }
    public function edit($id = null, $cid = null, $pcid = null)
    {
        $pageTitle = 'Edited Contents';
        $parentMenus = $this->menus->getparentMenu($id);
        $content = $this->contents->get($cid);
        $images = $this->contentImages->find('all')->where(['content_id' => $cid]);
        $subMenus = [];
        $content_sub_sub_menu_id = null;
        $content_sub_menu_id = null;

        if ($this->request->is(['post', 'put', 'patch'])) {

            $data = $this->request->getData();

            $data["status"] = 2;
            $data["content_id"] = $content->id;
            $sbmenu_id = $data["sbmenu_id"];
            $sbsbmenu_id = $data["sbsbmenu_id"];
            $sbsbsbmenu_id = $data["sbsbsbmenu_id"];
            $data["parent_menu_id"] = $data["pmenu_id"];
            $content = $this->contents->patchEntity($content, $data);
            $content->date_modified = FrozenTime::now();

            if ($this->contents->save($content)) {

                $pageContent = $this->pageContents->pageContents($data["content_id"]);

                if ($pageContent) {
                    if (!empty($sbsbmenu_id)) {
                        $pageContent->menu_id = $sbsbmenu_id;
                    } elseif (!empty($sbmenu_id)) {
                        $pageContent->menu_id = $sbmenu_id;
                    } else {
                        $pageContent->menu_id = $id;
                    }

                    $this->pageContents->save($pageContent);

                } else {
                    throw new NotFoundException(__('Page content not found.'));
                }

                //add images
                if (isset($data['form_key']) && $data['form_key'] < 0) {
                    $this->contentImages->updateAll(
                        ['content_id' => $content->id],
                        [
                            'content_id' => $data['form_key']
                        ]
                    );
                }

                $this->Flash->success(__('Content updated successfully.'));

                $section = $this->request->getParam('section');

                return $this->redirect([
                    'controller' => $section ? $this->request->getParam('section') :
                        $this->request->getParam('controller'),
                    'action' => 'editedListings'
                ]);
            }
        }

        // find the sub menu and sub sub menu of the selected content
        if ($pcid != null) {
            $selectedPage = ($pcid == null || $pcid == $id) ? null : $this->pageContents->find('all', [
                'conditions' => [
                    'PageContents.id' => $pcid
                ],
                'contain' => [
                    'Menus.ParentMenus'
                ]
            ])->first();

            if ($selectedPage != null && $selectedPage->menu != null && $selectedPage->menu->has('parent_menu')) {
                $parentMenu = $selectedPage->menu->parent_menu;

                if ($parentMenu->parent_menu_id != 0) {
                    $content_sub_menu_id = $parentMenu->id;
                    $content_sub_sub_menu_id = $selectedPage->menu_id;
                    $subMenus = $this->menus->getparentMenu($parentMenu->id);
                } else {
                    $content_sub_menu_id = $selectedPage->menu->id;
                    $subMenus = $this->menus->getparentMenu($selectedPage->menu_id);
                }
            } else {
                $content_sub_menu_id = $pcid;
            }
        }

        $menu_id = "";
        $this->set(compact('pageTitle', 'parentMenus', 'content', 'images', 'menu_id', 'content_sub_menu_id', 'content_sub_sub_menu_id', 'subMenus'));
    }

    public function addImage($id = null)
    {
        // $this->autoRender = false;

        if ($this->request->is(['post'])) {
            $file = $this->request->getData('file');
            // Check if the file was uploaded successfully
            if ($file && $file->getError() === UPLOAD_ERR_OK) {

                $filename = $file->getClientFilename();

                // Attempt to upload file
                $fileOK = $this->uploadFiles(Configure::read('IMAGE_PREPATH'), $file);
                if (!empty($fileOK['urls']) && !empty($fileOK['file_path'][0])) {
                    // Create a new entity
                    $contentImage = $this->contentImages->newEmptyEntity();

                    // Set additional properties
                    $contentImage->content_id = $id;

                    $contentImage->image_path = substr($fileOK['file_path'][0], Configure::read('IMAGE_PRELENGTH'), strlen($fileOK['file_path'][0]));


                    // Save entity to database
                    if ($this->contentImages->save($contentImage)) {
                        $this->log('Successfully uploaded and saved image', 'debug');
                    }

                }
            }

        }
        $this->set(compact('id'));
    }


    /**
     * Remove an image based on the provided ID.
     *
     * @param int|null $id The ID of the image to remove
     * @return \Cake\Http\Response|null Returns a JSON response indicating success or failure
     */
    public function removeImage($id = null)
    {
        $this->autoRender = false;
        //  $this->viewBuilder()->setLayout();

        $this->log('Ajax request received for Image removal', 'debug');
        $imageModel = $this->ContentImages;
        $imageItem = $imageModel->get($id);
        $this->log('image id to remove: ' . $imageItem->image_path, 'debug');
        $status = "Fail";

        if ($imageModel->delete($imageItem)) {
            $this->response = $this->response->withStatus(200);
            $status = "Success";
        } else {
            $this->response = $this->response->withStatus(500);
        }

        echo json_encode(array($status));
    }

    public function enableListing($id = null)
    {
        // Check if ID is provided
        if ($id === null) {
            $this->Flash->error(__('Invalid content ID.'));
            return $this->redirect($this->referer());
        }
        $result = $this->contents->updateAll(['status' => 1], ['id' => $id]);

        // Redirect to the referring page
        return $this->redirect($this->referer());
    }

    /**
     * Disable listing for a specific content.
     *
     * @param int|null $id Content ID
     * @return \Cake\Http\Response|null Redirects to the referring page
     */
    public function disableListing($id = null)
    {
        // Check if ID is provided
        if ($id === null) {
            $this->Flash->error(__('Invalid content ID.'));
            return $this->redirect($this->referer());
        }

        try {
            // Attempt to update content status
            $result = $this->contents->updateAll(['status' => 99], ['id' => $id]);

            // Check if any rows were updated
            if ($result) {
                $this->Flash->success(__('Content has been DISABLED successfully.'));
                Log::info('Content ID ' . $id . ' has been disabled successfully.');
            } else {
                throw new NotFoundException(__('Content not found or already disabled.'));
            }
        } catch (Exception $e) {
            // Log the error and set flash message
            Log::error('Error disabling content ID ' . $id . ': ' . $e->getMessage());
            $this->Flash->error(__('An error occurred while disabling the content.'));
        }

        // Redirect to the referring page
        return $this->redirect($this->referer());
    }

    public function showOnHome($id = null)
    {
        // Check if ID is provided
        if ($id === null) {
            $this->Flash->error(__('Invalid content ID.'));
            return $this->redirect($this->referer());
        }

        try {
            // Attempt to update the show_on_home status
            $result = $this->contents->updateAll(['show_on_home' => 1], ['id' => $id]);

            // Check if any rows were updated
            if ($result) {
                $this->Flash->success(__('Content successfully shown on homepage.'));
                Log::info('Content ID ' . $id . ' is now shown on the homepage.');
            } else {
                throw new NotFoundException(__('Content not found or already shown on the homepage.'));
            }
        } catch (Exception $e) {
            // Log the error and set flash message
            Log::error('Error showing content ID ' . $id . ' on homepage: ' . $e->getMessage());
            $this->Flash->error(__('An error occurred while showing the content on the homepage.'));
        }

        // Redirect to the referring page
        return $this->redirect($this->referer());
    }

    /* Remove specific content from the homepage.
     *
     * @param int|null $id Content ID
     * @return \Cake\Http\Response|null Redirects to the referring page
     */
    public function removeFromHome($id = null)
    {
        // Check if ID is provided
        if ($id === null) {
            $this->Flash->error(__('Invalid content ID.'));
            return $this->redirect($this->referer());
        }

        try {
            // Attempt to update the show_on_home status
            $result = $this->contents->updateAll(['show_on_home' => 0], ['id' => $id]);

            // Check if any rows were updated
            if ($result) {
                $this->Flash->success(__('Content removed from the homepage.'));
                Log::info('Content ID ' . $id . ' has been removed from the homepage.');
            } else {
                throw new NotFoundException(__('Content not found or already removed from the homepage.'));
            }
        } catch (Exception $e) {
            // Log the error and set flash message
            Log::error('Error removing content ID ' . $id . ' from homepage: ' . $e->getMessage());
            $this->Flash->error(__('An error occurred while removing the content from the homepage.'));
        }

        // Redirect to the referring page
        return $this->redirect($this->referer());
    }

    public function searchFilters($query)
    {
        $searchTerm = $this->request->getQuery('search');

        if ($searchTerm) {
            $query->matching('Contents', function ($q) use ($searchTerm) {
                return $this->contents->applySearchFilter($q, $searchTerm);
            });
        }

        return $query;
    }

}